#include "ToolConfig.h"

#include <QRadioButton>
#include <QMessageBox>

#include "ToolSpecificConfig.h"

#include "../../application/Reader.h"
#include "../../models/data/ModelData.h"
#include "../../models/data/ReaderEntry.h"

#include <QrfeReaderInterfaceGlobal>
#include <QrfeReaderInterface>

ToolConfig::ToolConfig(QPixmap icon, QString name, QString description, ConfigType type, ModelData* modelData, QWidget *parent)
	: QDialog(parent)
	, m_icon(icon)
	, m_name(name)
	, m_description(description)
	, m_type(type)
	, m_modelData(modelData)
	, m_specificConfig(0)
{
	ui.setupUi(this);
	buildUpUi();
}

ToolConfig::ToolConfig(QPixmap icon, QString name, QString description, ConfigType type, ModelData* modelData, ToolSpecificConfig* specificConfig, QWidget *parent)
    : QDialog(parent)
	, m_icon(icon)
	, m_name(name)
	, m_description(description)
	, m_type(type)
	, m_modelData(modelData)
	, m_specificConfig(specificConfig)
{
	ui.setupUi(this);
	buildUpUi();
}

ToolConfig::~ToolConfig()
{
	if(m_specificConfig != 0)
		delete m_specificConfig;
}


QList<QrfeReaderInterface*>  ToolConfig::readerList()
{
	return m_readerList;
}

QList<QString> 	ToolConfig::tagList()
{
	return m_tagList;
}


void ToolConfig::buildUpUi()
{
	QObject::connect(ui.startButton,		SIGNAL(clicked()), 			this, SLOT(tryAccept()));
	QObject::connect(ui.closeButton, 		SIGNAL(clicked()), 			this, SLOT(reject()));

	ui.iconLabel->setPixmap(m_icon);
	ui.nameLabel->setText(m_name);
	ui.descriptionLabel->setText(m_description);

	ui.readerSelectAllCheckBox->setVisible(false);
	ui.tagGroupBox->setVisible(false);
	ui.tagSelectAllCheckBox->setVisible(false);
	ui.settingsGroupBox->setVisible(false);

	ui.startButton->setEnabled(false);


	switch(m_type)
	{
	case SINGLE_READER:
		buildUpUi_SINGLE_READER();
		break;
	case MULTIPLE_READER:
		buildUpUi_MULTIPLE_READER();
		break;
	case SINGLE_READER_WITH_SINGLE_TAG:
	case SINGLE_READER_WITH_MULTIPLE_TAGS:
		buildUpUi_SINGLE_READER_WITH_TAG();
		break;
	}

	if(m_specificConfig != 0)
	{
		ui.settingsGroupBox->setVisible(true);
		ui.settingsLayout->addWidget(m_specificConfig->widget());
	}

	this->update();
	QTimer::singleShot(0, this, SLOT(shrink()));
}

void ToolConfig::buildUpUi_SINGLE_READER()
{
	QList<QString> reader = m_modelData->readerList();
	foreach(QString readerId, reader)
	{
		QrfeReaderInterface* interface = m_modelData->getReader(readerId)->readerInterface();
		if(interface->tagType() == QrfeGlobal::TAG_GEN2)
		{
			QRadioButton* b = new QRadioButton(interface->name(), ui.readerGroupBox);
			b->setProperty("ReaderId", readerId);
			ui.readerEntryLayout->addWidget(b);
			m_readerEntries << b;
		}
	}

	if(m_readerEntries.size() == 0)
	{
		QLabel* l = new QLabel("No Reader available", ui.readerGroupBox);
		ui.readerEntryLayout->addWidget(l);
	}
	else
	{
		if(qobject_cast<QRadioButton*>(m_readerEntries.at(0)) != 0)
			qobject_cast<QRadioButton*>(m_readerEntries.at(0))->click();

		ui.startButton->setEnabled(true);
	}
}

void ToolConfig::buildUpUi_MULTIPLE_READER()
{
	ui.readerSelectAllCheckBox->setVisible(true);

	QList<QString> reader = m_modelData->readerList();
	foreach(QString readerId, reader)
	{
		QrfeReaderInterface* interface = m_modelData->getReader(readerId)->readerInterface();
		if(interface->tagType() == QrfeGlobal::TAG_GEN2)
		{
			QCheckBox* b = new QCheckBox(interface->name(), ui.readerGroupBox);
			b->setProperty("ReaderId", readerId);
			ui.readerEntryLayout->addWidget(b);
			m_readerEntries << b;
		}
	}

	if(m_readerEntries.size() == 0)
	{
		QLabel* l = new QLabel("No Reader available", ui.readerGroupBox);
		ui.readerEntryLayout->addWidget(l);
	}
	else
	{
		if(qobject_cast<QCheckBox*>(m_readerEntries.at(0)) != 0)
			qobject_cast<QCheckBox*>(m_readerEntries.at(0))->click();

		if(m_readerEntries.size() > 1)
			ui.readerSelectAllCheckBox->setEnabled(true);
		else
			ui.readerSelectAllCheckBox->setEnabled(false);

		ui.startButton->setEnabled(true);
	}
}

void ToolConfig::buildUpUi_SINGLE_READER_WITH_TAG()
{
	QList<QString> reader = m_modelData->readerList();
	foreach(QString readerId, reader)
	{
		QrfeReaderInterface* interface = m_modelData->getReader(readerId)->readerInterface();
		if(interface->tagType() == QrfeGlobal::TAG_GEN2)
		{
			QAbstractButton* b = new QRadioButton(interface->name(), ui.readerGroupBox);
			b->setProperty("ReaderId", readerId);
			ui.readerEntryLayout->addWidget(b);
			m_readerEntries << b;
			connect(b, 		SIGNAL(toggled(bool)),
					this, 	  SLOT(singleReaderSelected(bool)));
		}
	}

	if(m_readerEntries.size() == 0)
	{
		QLabel* l = new QLabel("No Reader available", ui.readerGroupBox);
		ui.readerEntryLayout->addWidget(l);
	}
	else
	{
		if(m_readerEntries.at(0) != 0)
			m_readerEntries.at(0)->click();
		ui.tagGroupBox->setVisible(true);

		ui.startButton->setEnabled(true);
	}

}


void ToolConfig::tryAccept()
{
	m_readerList.clear();
	foreach(QAbstractButton* button, m_readerEntries)
	{
		if(button->isChecked())
		{
			QString readerId = button->property("ReaderId").toString();
			Reader* reader = m_modelData->getReader(readerId);
			if(!reader)
				continue;
			m_readerList << reader->readerInterface();
		}
	}

	m_tagList.clear();
	foreach(QAbstractButton* button, m_tagEntries)
	{
		if(button->isChecked())
			m_tagList << (button)->text();
	}

	if( m_specificConfig != 0 )
		if( m_specificConfig->checkInput() != true)
			return;

	QDialog::accept();
}

void ToolConfig::shrink()
{
#ifdef Q_OS_MACX
    this->resize(500,200);
#else
    this->resize(300,100);
#endif
	this->setParent(this->parentWidget());
}

void ToolConfig::singleReaderSelected(bool checked)
{
	QAbstractButton* b = qobject_cast<QAbstractButton*>(sender());
	if(!b)
		return;

	removeTagEntries();

	if(!checked && m_readerEntries.size() > 1){
		return;
	}
	else if(!checked && m_readerEntries.size() == 1){
		if(m_readerEntries.at(0) != 0)
			m_readerEntries.at(0)->click();
		return;
	}

	QString readerId = b->property("ReaderId").toString();

//	ReaderEntry* reader = m_modelData->getReader(readerId);
//	if(!reader)
//		return;

	QList<QString> tags = m_modelData->tagListOfReader(readerId);
	foreach(QString tag, tags)
	{
		QAbstractButton* button;
		if( m_type == SINGLE_READER_WITH_SINGLE_TAG )
			button = new QRadioButton(tag, ui.tagGroupBox);
		else if( m_type == SINGLE_READER_WITH_MULTIPLE_TAGS )
			button = new QCheckBox(tag, ui.tagGroupBox);

		connect(ui.tagSelectAllCheckBox, 	SIGNAL(toggled(bool)),
				button, 					  SLOT(setChecked(bool)) );

		ui.tagEntryLayout->addWidget(button);
		m_tagEntries << button;
	}

	ui.tagSelectAllCheckBox->setChecked(false);
	if( m_type == SINGLE_READER_WITH_MULTIPLE_TAGS && m_tagEntries.size() > 0)
	{
		ui.tagSelectAllCheckBox->setVisible(true);
		ui.tagSelectAllCheckBox->setChecked(true);
	}
	else if(m_type == SINGLE_READER_WITH_SINGLE_TAG && m_tagEntries.size() > 0)
	{
		if(m_tagEntries.at(0) != 0)
			m_tagEntries.at(0)->click();
	}

	this->update();
	QTimer::singleShot(0, this, SLOT(shrink()));
}




void ToolConfig::removeReaderEntries()
{
	foreach(QWidget* w, m_readerEntries){
		ui.readerEntryLayout->removeWidget(w);
		w->setVisible(false);
		delete w;
	}
	m_readerEntries.clear();
	ui.readerEntryLayout->update();

	this->repaint();
}

void ToolConfig::removeTagEntries()
{
	foreach(QWidget* w, m_tagEntries){
		ui.tagEntryLayout->removeWidget(w);
		w->setVisible(false);
		delete w;
	}
	m_tagEntries.clear();
	ui.tagEntryLayout->update();

	this->repaint();
}

